// DAZ Product Filter — content_v4.js (universalized)
// v4 UI (collapsible left box + force load overlay) + v3-style universal DOM parsing
// - Works on /shop pages AND promo/landing pages (e.g. /celebrate-the-new-year)
// - Applies Hide / Only Show / Highlight universally (no dependency on page_script.js)
// - Price filtering works where a price can be found in-card (falls back gracefully)
// - Still sends state to page_script.js if present (for pages where it provides extra features)

(() => {
  'use strict';

  // ============================================================================
  // ENV GUARD
  // ============================================================================

  if (typeof chrome === 'undefined' || !chrome.storage || !chrome.runtime) return;

  // Skip DAZ gallery entirely (keeps prior behavior)
  if (location.hostname === "www.daz3d.com" && location.pathname.startsWith("/gallery")) return;

  // ============================================================================
  // STORAGE KEYS
  // ============================================================================

  const MASTER_KEY = "daz_hider_master_enabled";
  const HIDE_KEY   = "daz_hider_rules";
  const HL_KEY     = "daz_hider_highlight_rules";
  const SHOW_KEY   = "daz_hider_show_rules";
  const PRICE_KEY  = "daz_hider_price_rules";
  const HIDE_COL   = "daz_hider_col_hide_enabled";
  const HL_COL     = "daz_hider_col_hl_enabled";
  const SHOW_COL   = "daz_hider_col_show_enabled";
  const PRICE_COL  = "daz_hider_col_price_enabled";

  // UI state (expanded/collapsed)
  const UI_EXPANDED_KEY = "daz_hider_ui_expanded";

  // ============================================================================
  // STATE
  // ============================================================================

  let state = {
    masterEnabled: true,
    rules: [],
    highlightRules: [],
    showOnlyRules: [],
    priceRules: { min: "", max: "", minEnabled: false, maxEnabled: false },
    colHide: true,
    colHL: true,
    colShow: true,
    colPrice: true
  };

  let currentCounts = { hidden: 0, highlighted: 0, shown: 0 };
  let isExpanded = true;
  let isForceLoading = false;

  // ============================================================================
  // SMALL HELPERS
  // ============================================================================

  const norm = (s) => (s || "").toString().trim().toLowerCase();

  function formatUSD(value) {
    const n = Number(String(value).replace(/[^0-9.]/g, ""));
    if (!Number.isFinite(n)) return null;
    return n.toLocaleString("en-US", { style: "currency", currency: "USD" });
  }

  function enabledNeedles(list, enabled) {
    if (!enabled) return [];
    if (!Array.isArray(list)) return [];
    return list
      .filter(r => r && r.enabled && r.text && r.text.trim())
      .map(r => norm(r.text));
  }

  const sleep = (ms) => new Promise(r => setTimeout(r, ms));

  // ============================================================================
  // UNIVERSAL PRODUCT DETECTION (from older v3 logic, slightly expanded)
  // ============================================================================

  // Primary: classic store tiles
  const ITEM_SEL_PRIMARY = "li[id^='product-'], li.item";
  // Fallback: generic product-ish blocks used across DAZ and promo pages
  const ITEM_SEL_FALLBACK = [
    // common product list structures
    "[data-product-id]",
    "article.product",
    ".product-card",
    ".product-item",
    ".product-tile",
    "[role='listitem']",
    ".slick-slide",
    ".product",
    // promo landing slabs (like /celebrate-the-new-year)
    "a.slab-link[data-title][href^='/']",
    ".slab-card",
    ".slab"
  ].join(",");

  const TITLE_SEL = [
    "[itemprop='name']",
    ".slab-link-title",
    "a.slab-link-title",
    ".slab-card__title",
    ".product-title",
    ".card-title",
    ".title",
    "h3",
    "h2",
    "h1",
    // fallback: a link that's not the image-only slab
    "a:not(.slab-link-img):not([href^='#'])"
  ].join(",");

  const PRICE_SELECTORS = [
    "[data-price]",
    "[class*='price']",
    "[class*='amount']",
    "[class*='cost']",
    ".cost",
    ".amount",
    ".price"
  ];

  const visible = (el) => {
    try {
      const r = el?.getBoundingClientRect?.();
      return !r || (r.width > 0 && r.height > 0);
    } catch { return true; }
  };

  const looksCard = (el) => {
    try {
      if (!el || typeof el.querySelector !== "function") return false;
      // heuristic: card has at least something card-like
      return !!el.querySelector("img,a,[class*='price'],[data-price],[class*='add'],button");
    } catch {
      return false;
    }
  };

  function cards(root = document) {
    let items = Array.from(root.querySelectorAll(ITEM_SEL_PRIMARY));
    if (!items.length) items = Array.from(root.querySelectorAll(ITEM_SEL_FALLBACK));
    // If we matched only the slab <a>, prefer the closest slab/card container if it exists.
    items = items.map(el => el.closest?.(".slab-card, .slab, article, li, div") || el);
    // de-dupe
    const out = [];
    const seen = new Set();
    for (const el of items) {
      if (!el || seen.has(el)) continue;
      seen.add(el);
      if (looksCard(el)) out.push(el);
    }
    return out;
  }

  function titleOf(card) {
    try {
      // Promo slab anchor often has data-title
      const slabLink = card.matches?.("a.slab-link[data-title]") ? card
        : card.querySelector?.("a.slab-link[data-title]");
      if (slabLink && slabLink.dataset && slabLink.dataset.title) {
        const t = (slabLink.dataset.title || "").trim();
        if (t) return t;
      }

      const cands = Array.from(card.querySelectorAll(TITLE_SEL)).filter(el => {
        const t = (el.textContent || "").trim();
        if (!t) return false;
        const tl = t.toLowerCase();
        if (tl.includes("add to cart")) return false;
        // avoid picking price strings as a title
        if (/\$\s*\d/.test(tl)) return false;
        return visible(el);
      });

      if (!cands.length) return null;
      let best = cands[0];
      for (const el of cands) {
        if (el.textContent.trim().length > best.textContent.trim().length) best = el;
      }
      return best.textContent.trim();
    } catch {
      return null;
    }
  }

  function priceOf(card) {
    try {
      // 1) Look for price-like elements inside the card
      for (const sel of PRICE_SELECTORS) {
        const els = Array.from(card.querySelectorAll(sel));
        for (const el of els) {
          // data-price
          const dp = el.getAttribute?.("data-price");
          if (dp != null && dp !== "") {
            const n = parseFloat(String(dp).replace(/[^0-9.]/g, ""));
            if (Number.isFinite(n) && n >= 0) return n;
          }
          // text-based
          const text = (el.textContent || "").trim();
          const m = text.match(/\$(\d+(?:\.\d{1,2})?)/);
          if (m) {
            const n = parseFloat(m[1]);
            if (Number.isFinite(n) && n >= 0) return n;
          }
        }
      }

      // 2) fallback: regex search in whole card text
      const cardText = (card.textContent || "").trim();
      const m2 = cardText.match(/\$(\d+(?:\.\d{1,2})?)/);
      if (m2) {
        const n = parseFloat(m2[1]);
        if (Number.isFinite(n) && n >= 0) return n;
      }
    } catch { /* ignore */ }

    return null;
  }

  function haystack(card) {
    try { return norm(card.textContent || ""); } catch { return ""; }
  }

  function gather(root = document) {
    const out = [];
    const seen = new Set();

    for (const it of cards(root)) {
      if (seen.has(it)) continue;
      const t = titleOf(it);
      if (!t) continue;
      const p = priceOf(it);
      out.push({ container: it, titleText: t, searchText: haystack(it), price: p });
      seen.add(it);
    }

    return out;
  }

  // ============================================================================
  // HIDE/RESTORE MECHANISM (comment placeholder, like older v3)
  // ============================================================================

  const nodeToPH = new WeakMap(); // node -> comment placeholder
  const phToNode = new WeakMap(); // placeholder -> node
  const phSet = new Set();        // placeholders currently in DOM
  const hidden = new WeakMap();   // node -> boolean hidden

  function removeNode(node) {
    try {
      if (!node || nodeToPH.has(node)) return;
      const p = node.parentNode;
      if (!p) return;
      const ph = document.createComment("daz-hider");
      p.replaceChild(ph, node);
      nodeToPH.set(node, ph);
      phToNode.set(ph, node);
      phSet.add(ph);
      hidden.set(node, true);
    } catch { /* ignore */ }
  }

  function restoreNode(node) {
    try {
      const ph = nodeToPH.get(node);
      if (!ph) { hidden.set(node, false); return; }
      const p = ph.parentNode;
      if (p) p.replaceChild(node, ph);
      nodeToPH.delete(node);
      phToNode.delete(ph);
      phSet.delete(ph);
      hidden.set(node, false);
    } catch { /* ignore */ }
  }

  function restoreAll() {
    for (const ph of Array.from(phSet)) {
      const n = phToNode.get(ph);
      if (n) restoreNode(n);
    }
  }

  // ============================================================================
  // HIGHLIGHT CLASS (CSS)
  // ============================================================================

  let styleTag;
  function ensureStyle() {
    if (styleTag) return;
    styleTag = document.createElement("style");
    styleTag.textContent = `
      .daz-hl{
        outline:3px solid #00ff57!important;
        outline-offset:2px;
        border-radius:6px;
      }
    `;
    document.documentElement.appendChild(styleTag);
  }

  // ============================================================================
  // UNIVERSAL APPLY FILTERS
  // ============================================================================

  let timer = 0;
  let running = false;

  function scheduleApply() {
    if (timer) clearTimeout(timer);
    timer = setTimeout(applyFiltersUniversal, 110);
  }

  function applyFiltersUniversal() {
    if (running) return;
    running = true;

    try {
      ensureStyle();

      // If master is off, restore all + clear highlights
      if (!state.masterEnabled) {
        restoreAll();
        document.querySelectorAll(".daz-hl").forEach(el => el.classList.remove("daz-hl"));
        currentCounts = { hidden: 0, highlighted: 0, shown: productCount(), };
        updateFloatingBox();
        return;
      }

      const entries = gather();

      const hideNeedles = enabledNeedles(state.rules, state.colHide);
      const hlNeedles   = enabledNeedles(state.highlightRules, state.colHL);
      const showNeedles = enabledNeedles(state.showOnlyRules, state.colShow);
      const showMode = showNeedles.length > 0;

      // price bounds
      const pr = state.priceRules || {};
      const priceColOn = !!state.colPrice;

      const minOn = priceColOn && pr.minEnabled === true && pr.min !== "" && pr.min != null && !isNaN(parseFloat(pr.min));
      const maxOn = priceColOn && pr.maxEnabled === true && pr.max !== "" && pr.max != null && !isNaN(parseFloat(pr.max));

      const minVal = minOn ? parseFloat(pr.min) : null;
      const maxVal = maxOn ? parseFloat(pr.max) : null;

      const hideSet = new Set();
      const hlSet = new Set();

      for (const e of entries) {
        const t = norm(e.titleText);
        const hay = e.searchText || "";

        // Only show: allow match on title OR card text
        const matchShow = showMode && showNeedles.some(k => t.includes(k) || (hay && hay.includes(k)));

        // Hide: title match only (keeps behavior consistent with earlier v3)
        const matchHide = hideNeedles.some(k => t.includes(k));

        // Highlight: haystack match
        const matchHL = hlNeedles.length && hay && hlNeedles.some(k => hay.includes(k));

        // Price match: only applies if a bound is enabled AND a price is detected.
        // If price isn't detectable, we do NOT hide (graceful degrade for promo pages).
        let matchPrice = true;
        if (minOn || maxOn) {
          if (typeof e.price === "number" && Number.isFinite(e.price)) {
            if (minOn && e.price < minVal) matchPrice = false;
            if (maxOn && e.price > maxVal) matchPrice = false;
          }
        }

        // Final hide decision
        let wantHide;
        if (showMode) {
          wantHide = !matchShow;
        } else {
          wantHide = matchHide;
        }

        // Apply price constraint after show/hide logic (if it fails, hide it)
        if (!wantHide && (minOn || maxOn) && matchPrice === false) {
          wantHide = true;
        }

        if (wantHide) hideSet.add(e.container);
        else if (matchHL) hlSet.add(e.container);
      }

      // Apply hide/restore
      for (const e of entries) {
        const c = e.container;
        const want = hideSet.has(c);
        const isH = hidden.get(c) === true;
        if (want && !isH) removeNode(c);
        else if (!want && isH) restoreNode(c);
      }

      // Apply highlights
      let highlighted = 0;
      for (const e of entries) {
        const c = e.container;
        if (hidden.get(c) === true) { try { c.classList.remove("daz-hl"); } catch {} continue; }
        if (hlSet.has(c)) { try { c.classList.add("daz-hl"); } catch {} highlighted++; }
        else { try { c.classList.remove("daz-hl"); } catch {} }
      }

const hiddenCount = phSet.size;                 // <- true hidden count
const visibleCount = entries.length;            // visible entries still in DOM
const total = visibleCount + hiddenCount;       // best estimate of total
const shown = visibleCount;

currentCounts = { hidden: hiddenCount, highlighted, shown };

      updateFloatingBox();

      // Also inform page_script (if listening) to keep it in sync
      window.postMessage({ type: 'DAZ_HIDER_UPDATE_COUNTS', counts: currentCounts }, '*');

    } finally {
      running = false;
    }
  }

  function productCount() {
    try { return cards().length; } catch { return 0; }
  }

  // ============================================================================
  // STATE MANAGEMENT
  // ============================================================================

  function loadState(callback) {
    const keys = {
      [MASTER_KEY]: true,
      [HIDE_KEY]: [],
      [HL_KEY]: [],
      [SHOW_KEY]: [],
      [PRICE_KEY]: { min: "", max: "", minEnabled: false, maxEnabled: false },
      [HIDE_COL]: true,
      [HL_COL]: true,
      [SHOW_COL]: true,
      [PRICE_COL]: true,
      [UI_EXPANDED_KEY]: true  // first run defaults to expanded
    };

    chrome.storage.sync.get(keys, (result) => {
      state.masterEnabled = result[MASTER_KEY] !== false;
      state.rules = result[HIDE_KEY] || [];
      state.highlightRules = result[HL_KEY] || [];
      state.showOnlyRules = result[SHOW_KEY] || [];
      state.priceRules = result[PRICE_KEY] || { min: "", max: "", minEnabled: false, maxEnabled: false };
      state.colHide = result[HIDE_COL] !== false;
      state.colHL = result[HL_COL] !== false;
      state.colShow = result[SHOW_COL] !== false;
      state.colPrice = result[PRICE_COL] !== false;

      isExpanded = result[UI_EXPANDED_KEY] !== false;

      if (callback) callback();
    });
  }

  chrome.storage.onChanged.addListener((changes, area) => {
    if (area !== 'sync') return;
    loadState(() => {
      sendStateToPageScript();
      updateFloatingBox();
      scheduleApply();
    });
  });

chrome.runtime.onMessage.addListener((msg) => {
  if (!msg) return;

  if (msg.type === "STATE_UPDATED") {
    loadState(() => {
      sendStateToPageScript();
      updateFloatingBox();
      scheduleApply();
    });
    return;
  }

  // Manual hard refresh (mimics your off/on fix without changing master state)
  if (msg.type === "FORCE_REFRESH") {
    try { restoreAll(); } catch {}
    try { document.querySelectorAll(".daz-hl").forEach(el => el.classList.remove("daz-hl")); } catch {}
    scheduleApply();
    return;
  }
});


  // ============================================================================
  // COMMUNICATION WITH PAGE SCRIPT (optional / compatibility)
  // ============================================================================

  function sendStateToPageScript() {
    window.postMessage({ type: 'DAZ_HIDER_STATE_UPDATE', state }, '*');
  }

  window.addEventListener('message', (event) => {
    if (event.source !== window) return;

    // Keep compatibility if page_script still emits counts
    if (event.data?.type === 'DAZ_HIDER_UPDATE_COUNTS' && event.data.counts) {
      // If page_script is driving counts, accept them, but our universal loop
      // will also refresh counts on the next schedule.
      currentCounts = event.data.counts;
      updateFloatingBox();
      return;
    }

    if (event.data?.type === 'DAZ_HIDER_OPEN_POPUP') {
      openPopup();
    }

    // If page script requests manual refresh, run universal apply too
    if (event.data?.type === 'DAZ_HIDER_MANUAL_REFRESH') {
      scheduleApply();
    }
  });

  // ============================================================================
  // POPUP CONTROL
  // ============================================================================

function safeSendMessage(message, cb) {
  try {
    // If the extension context is gone, runtime/id may be unavailable
    if (!chrome?.runtime?.id) return;

    chrome.runtime.sendMessage(message, (resp) => {
      // In MV3 it's normal to get lastError when context is changing or SW isn't reachable
      void chrome.runtime.lastError;
      cb?.(resp);
    });
  } catch (e) {
    // Handles: "Extension context invalidated."
    // Intentionally ignore.
  }
}

function openPopup() {
  safeSendMessage({ type: "OPEN_POPUP" });
}


  // ============================================================================
  // FORCE LOAD (scrolling) — now uses universal productCount()
  // ============================================================================

  async function forceLoadProducts() {
    if (isForceLoading) return;
    isForceLoading = true;

    let cancelRequested = false;
    const overlay = createOverlay();
    overlay.style.display = 'flex';

    const cancelBtn = overlay.querySelector('#daz-cancel-btn');
    if (cancelBtn) {
      cancelBtn.addEventListener('click', () => {
        cancelRequested = true;
        cancelBtn.textContent = 'CANCELLING...';
        cancelBtn.style.opacity = '0.5';
        cancelBtn.style.pointerEvents = 'none';
      }, { once: true });
    }

    const subEl = overlay.querySelector('.overlay-sub');

    try {
      const scroller = document.scrollingElement || document.documentElement || document.body;
      const startY = (typeof window.scrollY === 'number') ? window.scrollY : scroller.scrollTop || 0;
      const step = Math.max(120, Math.floor((window.innerHeight || 800) * 0.75));

      window.scrollTo({ top: 0, behavior: 'auto' });
      await sleep(60);

      let lastH = 0, lastN = 0, idle = 0;
      const stableRounds = 6;
      const maxMs = 20000;
      const t0 = performance.now();

      while (performance.now() - t0 < maxMs) {
        if (cancelRequested) break;

        const target = Math.min(
          scroller.scrollTop + step,
          Math.max(0, scroller.scrollHeight - scroller.clientHeight)
        );

        window.scrollTo({ top: target, behavior: 'auto' });
        await new Promise(r => requestAnimationFrame(r));
        await sleep(24);

        const h = scroller.scrollHeight;
        const n = productCount();
        const atBottom = scroller.scrollTop + scroller.clientHeight >= h - 2;

        if (subEl) subEl.textContent = `Loaded items: ${n}`;

        if (atBottom) {
          if (h <= lastH + 1 && n <= lastN) idle++;
          else idle = 0;
          if (idle >= stableRounds) break;
        }

        lastH = Math.max(lastH, h);
        lastN = Math.max(lastN, n);
      }

      window.scrollTo({ top: startY, behavior: 'auto' });

      // Trigger both page_script refresh + our universal refresh
      window.postMessage({ type: 'DAZ_HIDER_MANUAL_REFRESH' }, '*');
      scheduleApply();

    } finally {
      isForceLoading = false;
      overlay.style.display = 'none';
    }
  }

  function createOverlay() {
    let oldOverlay = document.getElementById('daz-hider-overlay');
    if (oldOverlay) oldOverlay.remove();

    const overlay = document.createElement('div');
    overlay.id = 'daz-hider-overlay';
    overlay.style.cssText = `
      position: fixed;
      inset: 0;
      background: rgba(0, 0, 0, 0.95);
      color: #fff;
      display: none;
      align-items: center;
      justify-content: center;
      z-index: 9999999;
      font-family: system-ui, -apple-system, sans-serif;
    `;

overlay.innerHTML = `
  <div style="text-align: center; padding: 24px; max-width: 520px;">
    <div class="overlay-msg" style="font-size: 24px; font-weight: 700;">
      FORCE LOADING... <span style="font-weight: 500; font-size: 16px;">(for approx 20 seconds)</span>
    </div>

    <div class="overlay-sub" style="font-size: 17.5px; color: #eee; margin-top: 10px;">
      Loaded items: 0
    </div>

    <div id="daz-cancel-btn" style="
      margin-top: 22px;
      padding: 12px 24px;
      background: #ff4444;
      color: white;
      border-radius: 999px;
      cursor: pointer;
      font-size: 14px;
      font-weight: 700;
      display: inline-block;
      box-shadow: 0 2px 8px rgba(0,0,0,0.4);
      user-select: none;
    ">CANCEL REFRESH</div>

    <div style="margin-top: 14px; font-size: 13px; color: #cfcfcf; line-height: 1.35;">
      You can click <b>“Force Load”</b> as often as you'd like to load more products.
    </div>
  </div>
`;

    document.body.appendChild(overlay);
    return overlay;
  }

  // ============================================================================
  // COLLAPSIBLE FLOATING BOX (Left Side)
  // ============================================================================

  function createFloatingBox() {
    const container = document.createElement('div');
    container.id = 'daz-hider-floating-box';
    container.style.cssText = `
      position: fixed;
      top: 120px;
      left: 10px;
      z-index: 999999;
      display: flex;
      align-items: center;
    `;

    const toggleBtn = document.createElement('div');
    toggleBtn.id = 'daz-hider-toggle-btn';
    toggleBtn.style.cssText = `
      width: 50px;
      height: 50px;
      background: #ff4444;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      box-shadow: 0 2px 8px rgba(0,0,0,0.3);
      font-size: 28px;
      font-weight: 900;
      color: white;
      user-select: none;
      z-index: 2;
      position: relative;
      transition: transform 0.1s;
    `;
    toggleBtn.textContent = isExpanded ? '◀' : '▶';

    toggleBtn.addEventListener('mousedown', () => { toggleBtn.style.transform = 'scale(0.95)'; });
    toggleBtn.addEventListener('mouseup',   () => { toggleBtn.style.transform = 'scale(1)'; });

    toggleBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      toggleBox();
    });

    const infoBox = document.createElement('div');
    infoBox.id = 'daz-hider-info-box';
    infoBox.style.cssText = `
      background: #4444ff;
      color: white;
      padding: 12px 12px 12px 60px;
      border-radius: 8px;
      margin-left: -50px;
      box-shadow: 0 2px 8px rgba(0,0,0,0.3);
      font-family: system-ui, -apple-system, sans-serif;
      font-size: 14px;
      font-weight: 600;
      cursor: pointer;
      user-select: none;
      min-width: 180px;
      transition: all 0.3s ease;
      opacity: 1;
      transform: translateX(0);
    `;

    infoBox.innerHTML = `
      <div id="daz-master-toggle" style="
        margin-bottom: 12px;
        padding: 8px;
        border-radius: 4px;
        text-align: center;
        cursor: pointer;
        font-size: 16px;
        font-weight: bold;
        transition: all 0.2s ease;
      ">ENABLED (on/off)</div>

      <div style="margin-bottom: 6px;">Hidden: <span id="daz-hidden-count">0</span></div>
      <div style="margin-bottom: 6px;">Shown: <span id="daz-shown-count">0</span></div>
      <div style="margin-bottom: 6px;">Highlighted: <span id="daz-highlighted-count">0</span></div>

      <div id="daz-price-line" style="margin-top: 8px; display: none;"></div>

      <div id="daz-force-load-btn" style="
        margin-top: 10px;
        padding: 8px;
        background: #ff8800;
        border-radius: 4px;
        text-align: center;
        cursor: pointer;
        font-size: 12px;
        display: none;
      ">"ONLY SHOW" - Force Load (20 secs)</div>

      <div id="daz-settings-btn" style="
        margin-top: 16px;
        padding: 8px;
        background: #00cc00;
        border-radius: 4px;
        text-align: center;
        cursor: pointer;
        font-size: 12px;
        color: white;
        font-weight: 600;
      ">SETTINGS</div>
    `;

    if (!isExpanded) {
      infoBox.style.opacity = '0';
      infoBox.style.transform = 'translateX(-20px)';
      infoBox.style.pointerEvents = 'none';
    }

    infoBox.addEventListener('click', (e) => {
      console.log('[DAZ Hider] InfoBox clicked, target:', e.target.id, e.target);

      if (e.target.id === 'daz-master-toggle' || e.target.closest('#daz-master-toggle')) {
        console.log('[DAZ Hider] 🔄 Master toggle clicked!');
        e.stopPropagation();
        e.preventDefault();

        const newValue = !state.masterEnabled;
        console.log('[DAZ Hider] Master enabled now:', newValue);

        // Save to storage - this will trigger the storage listener which handles everything
        chrome.storage.sync.set({ daz_hider_master_enabled: newValue }, () => {
          console.log('[DAZ Hider] ✅ Saved master toggle to storage');
          // Update local state after save
          state.masterEnabled = newValue;
          // Update UI immediately
          updateFloatingBox();
        });

        return;
      }
      if (e.target.id === 'daz-force-load-btn' || e.target.closest('#daz-force-load-btn')) {
        e.stopPropagation();
        forceLoadProducts();
        return;
      }
      if (e.target.id === 'daz-settings-btn' || e.target.closest('#daz-settings-btn')) {
        e.stopPropagation();
        openPopup();
        return;
      }
      openPopup();
    });

    container.appendChild(toggleBtn);
    container.appendChild(infoBox);
    document.body.appendChild(container);

    updateFloatingBox();
  }

  function toggleBox() {
    isExpanded = !isExpanded;
    chrome.storage.sync.set({ [UI_EXPANDED_KEY]: isExpanded });

    const toggleBtn = document.getElementById('daz-hider-toggle-btn');
    const infoBox = document.getElementById('daz-hider-info-box');

    if (!toggleBtn || !infoBox) return;

    if (isExpanded) {
      toggleBtn.textContent = '◀';
      infoBox.style.opacity = '1';
      infoBox.style.transform = 'translateX(0)';
      infoBox.style.pointerEvents = 'auto';
    } else {
      toggleBtn.textContent = '▶';
      infoBox.style.opacity = '0';
      infoBox.style.transform = 'translateX(-20px)';
      infoBox.style.pointerEvents = 'none';
    }
  }

  function updateFloatingBox() {
    const masterToggle = document.getElementById('daz-master-toggle');
    const hiddenCount = document.getElementById('daz-hidden-count');
    const shownCount = document.getElementById('daz-shown-count');
    const highlightedCount = document.getElementById('daz-highlighted-count');
    const forceLoadBtn = document.getElementById('daz-force-load-btn');
    const priceLineEl = document.getElementById('daz-price-line');

    // Update master toggle (only if changed to prevent loops)
    if (masterToggle) {
      const expectedText = state.masterEnabled ? 'ENABLED (on/off)' : 'DISABLED (on/off)';
      const expectedBg = state.masterEnabled ? '#00aa00' : '#cc0000';

      // Check if BOTH text and background need updating
      if (masterToggle.textContent !== expectedText || masterToggle.style.background !== expectedBg) {
        console.log('[DAZ Hider] Updating master toggle, enabled:', state.masterEnabled);
        masterToggle.textContent = expectedText;
        masterToggle.style.background = expectedBg;
        masterToggle.style.color = 'white';
      }
    }

    if (hiddenCount) hiddenCount.textContent = currentCounts.hidden;
    if (shownCount) shownCount.textContent = currentCounts.shown;
    if (highlightedCount) highlightedCount.textContent = currentCounts.highlighted;

    // Show force load button only when "Only Show" is active
    if (forceLoadBtn) {
      const hasShowOnlyRules = !!state.colShow && Array.isArray(state.showOnlyRules) &&
        state.showOnlyRules.some(r => r && r.enabled && r.text && r.text.trim());
      forceLoadBtn.style.display = hasShowOnlyRules ? 'block' : 'none';
    }

    // Price line (respect column ON/OFF + individual toggles)
    if (priceLineEl) {
      if (!state.colPrice) {
        priceLineEl.textContent = "";
        priceLineEl.style.display = "none";
      } else {
        const pr = state.priceRules || {};
        const minOn = pr.minEnabled === true;
        const maxOn = pr.maxEnabled === true;

        const minFormatted = (minOn && pr.min !== "" && pr.min != null) ? formatUSD(pr.min) : null;
        const maxFormatted = (maxOn && pr.max !== "" && pr.max != null) ? formatUSD(pr.max) : null;

        if (!minFormatted && !maxFormatted) {
          priceLineEl.textContent = "";
          priceLineEl.style.display = "none";
        } else {
          const parts = [];
          if (minFormatted) parts.push(`${minFormatted} min`);
          if (maxFormatted) parts.push(`${maxFormatted} max`);
          priceLineEl.textContent = `Price: ${parts.join(", ")}`;
          priceLineEl.style.display = "block";
        }
      }
    }
  }

  // ============================================================================
  // INIT + OBSERVERS
  // ============================================================================

  function startObservers() {
    // Mutation observer: rerun apply on DOM changes (promo pages are often JS-rendered)
    const mo = new MutationObserver(() => scheduleApply());
    mo.observe(document.documentElement, { childList: true, subtree: true });

    // Navigation changes (some DAZ pages use SPA-ish transitions)
    window.addEventListener("popstate", scheduleApply);
    window.addEventListener("hashchange", scheduleApply);

    // Some pages lazy-load on scroll; light hook
    window.addEventListener("scroll", () => scheduleApply(), { passive: true });
  }

  function injectPageScript() {
    const script = document.createElement('script');
    script.src = chrome.runtime.getURL('page_script.js');
    script.onload = () => { try { script.remove(); } catch {} };
    (document.head || document.documentElement).appendChild(script);
  }

  function init() {
    loadState(() => {
      const waitForBody = setInterval(() => {
        if (!document.body) return;
        clearInterval(waitForBody);

        createFloatingBox();
        updateFloatingBox(); // Initialize toggle state immediately
        injectPageScript();

        // send state to page script after injection
        setTimeout(() => sendStateToPageScript(), 400);

        // start universal filtering
        startObservers();
        scheduleApply();
      }, 10);
    });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
